/*
   --------------------------------------------------------------------------------------------------------------------
   Example sketch/program showing An Arduino Door Access Control featuring RFID, EEPROM, Relay
   --------------------------------------------------------------------------------------------------------------------
   This is a MFRC522 library example; for further details and other examples see: https://github.com/miguelbalboa/rfid

   This example showing a complete Door Access Control System

  Simple Work Flow (not limited to) :
                                     +---------+
  +----------------------------------->READ TAGS+^------------------------------------------+
  |                              +--------------------+                                     |
  |                              |                    |                                     |
  |                              |                    |                                     |
  |                         +----v-----+        +-----v----+                                |
  |                         |MASTER TAG|        |OTHER TAGS|                                |
  |                         +--+-------+        ++-------------+                            |
  |                            |                 |             |                            |
  |                            |                 |             |                            |
  |                      +-----v---+        +----v----+   +----v------+                     |
  |         +------------+READ TAGS+---+    |KNOWN TAG|   |UNKNOWN TAG|                     |
  |         |            +-+-------+   |    +-----------+ +------------------+              |
  |         |              |           |                |                    |              |
  |    +----v-----+   +----v----+   +--v--------+     +-v----------+  +------v----+         |
  |    |MASTER TAG|   |KNOWN TAG|   |UNKNOWN TAG|     |GRANT ACCESS|  |DENY ACCESS|         |
  |    +----------+   +---+-----+   +-----+-----+     +-----+------+  +-----+-----+         |
  |                       |               |                 |               |               |
  |       +----+     +----v------+     +--v---+             |               +--------------->
  +-------+EXIT|     |DELETE FROM|     |ADD TO|             |                               |
          +----+     |  EEPROM   |     |EEPROM|             |                               |
                     +-----------+     +------+             +-------------------------------+


   Use a Master Card which is act as Programmer then you can able to choose card holders who will granted access or not

 * **Easy User Interface**

   Just one RFID tag needed whether Delete or Add Tags. You can choose to use Leds for output or Serial LCD module to inform users.

 * **Stores Information on EEPROM**

   Information stored on non volatile Arduino's EEPROM memory to preserve Users' tag and Master Card. No Information lost
   if power lost. EEPROM has unlimited Read cycle but roughly 100,000 limited Write cycle.

 * **Security**
   To keep it simple we are going to use Tag's Unique IDs. It's simple and not hacker proof.

   @license Released into the public domain.

   Typical pin layout used:
   -----------------------------------------------------------------------------------------
               MFRC522      Arduino       Arduino   Arduino    Arduino          Arduino
               Reader/PCD   Uno/101       Mega      Nano v3    Leonardo/Micro   Pro Micro
   Signal      Pin          Pin           Pin       Pin        Pin              Pin
   -----------------------------------------------------------------------------------------
   RST/Reset   RST          9             5         D9         RESET/ICSP-5     RST
   SPI SS      SDA(SS)      10            53        D10        10               10
   SPI MOSI    MOSI         11 / ICSP-4   51        D11        ICSP-4           16
   SPI MISO    MISO         12 / ICSP-1   50        D12        ICSP-1           14
   SPI SCK     SCK          13 / ICSP-3   52        D13        ICSP-3           15
*/



#include <EEPROM.h>     // We are going to read and write PICC's UIDs from/to EEPROM
#include <SPI.h>        // RC522 Module uses SPI protocol
#include <MFRC522.h>  // Library for Mifare RC522 Devices

/*
   define Sonar
*/

#define sonarTrigPin 7
#define sonarEchoPin 8

/*
  Instead of a Relay you may want to use a servo. Servos can lock and unlock door locks too
  Relay will be used by default
*/

#include <Servo.h>
Servo myservo;

/*
  Adding the addressable LEDs and library required
*/

#include <Adafruit_NeoPixel.h>
#ifdef __AVR__
#include <avr/power.h>
#endif

// Which pin on the Arduino is connected to the NeoPixels?
#define PIN            6

// How many NeoPixels are attached to the Arduino?
#define NUMPIXELS      9

// When we setup the NeoPixel library, we tell it how many pixels, and which pin to use to send signals.
Adafruit_NeoPixel pixels = Adafruit_NeoPixel(NUMPIXELS, PIN, NEO_GRB + NEO_KHZ800);

/*
  For visualizing whats going on hardware we need some leds and to control door lock a relay and a wipe button
  (or some other hardware) Used common anode led,digitalWriting HIGH turns OFF led Mind that if you are going
  to use common cathode led or just seperate leds, simply comment out #define COMMON_ANODE,
*/

#define COMMON_ANODE

#ifdef COMMON_ANODE
#define LED_ON LOW
#define LED_OFF HIGH
#else
#define LED_ON HIGH
#define LED_OFF LOW
#endif

int posServo = 0; // set pos Servo by default
int switchState = 0; // set state for button detecting the presence of object
//constexpr uint8_t redLed = 7;   // Set Led Pins
//constexpr uint8_t greenLed = 0;
//constexpr uint8_t blueLed = 5;

//constexpr uint8_t relay = 4;     // Set Relay Pin
constexpr uint8_t wipeB = 4;     // Button pin for WipeMode

bool programMode = false;  // initialize programming mode to false

uint8_t successRead;    // Variable integer to keep if we have Successful Read from Reader

byte storedCard[4];   // Stores an ID read from EEPROM
byte readCard[4];   // Stores scanned ID read from RFID Module
byte masterCard[4];   // Stores master card's ID read from EEPROM

// Create MFRC522 instance.
constexpr uint8_t RST_PIN = 9;     // Configurable, see typical pin layout above
constexpr uint8_t SS_PIN = 10;     // Configurable, see typical pin layout above

MFRC522 mfrc522(SS_PIN, RST_PIN);

///////////////////////////////////////// Setup ///////////////////////////////////
void setup() {

  //Sonar Configuration
  pinMode(sonarTrigPin, OUTPUT);
  pinMode(sonarEchoPin, INPUT);

  //Adafruit Pixel Configuration
  pixels.begin(); // This initializes the NeoPixel library.

  //Servo Configuration
  pinMode(2, INPUT);
  myservo.attach(3);

  //Arduino Pin Configuration
  //pinMode(redLed, OUTPUT);
  //pinMode(greenLed, OUTPUT);
  //pinMode(blueLed, OUTPUT);
  pinMode(wipeB, INPUT);   // Enable pin's pull up resistor
  //  pinMode(relay, OUTPUT);

  //Be careful how relay circuit behave on while resetting or power-cycling your Arduino
  //  digitalWrite(relay, HIGH);    // Make sure door is locked
  //digitalWrite(redLed, LED_OFF);  // Make sure led is off
  //digitalWrite(greenLed, LED_OFF);  // Make sure led is off
  //digitalWrite(blueLed, LED_OFF); // Make sure led is off
  pixels.setPixelColor(0, pixels.Color(0, 0, 0)); // LED 0 is off.
  pixels.show(); // This sends the updated pixel color to the hardware.

  //Protocol Configuration
  Serial.begin(9600);  // Initialize serial communications with PC
  SPI.begin();           // MFRC522 Hardware uses SPI protocol
  mfrc522.PCD_Init();    // Initialize MFRC522 Hardware

  //If you set Antenna Gain to Max it will increase reading distance
  //mfrc522.PCD_SetAntennaGain(mfrc522.RxGain_max);

  Serial.println(F("Access Control Example v0.1"));   // For debugging purposes
  ShowReaderDetails();  // Show details of PCD - MFRC522 Card Reader details

  //Wipe Code - If the Button (wipeB) Pressed while setup run (powered on) it wipes EEPROM
  if (digitalRead(wipeB) == HIGH) {  // when button pressed pin should get low, button connected to ground
    //digitalWrite(redLed, LED_ON); // Red Led stays on to inform user we are going to wipe
    pixels.setPixelColor(0, pixels.Color(220, 0, 0)); // Red color.
    pixels.show(); // This sends the updated pixel color to the hardware.
    Serial.println(F("Wipe Button Pressed"));
    Serial.println(F("You have 10 seconds to Cancel"));
    Serial.println(F("This will be remove all records and cannot be undone"));
    bool buttonState = monitorWipeButton(10000); // Give user enough time to cancel operation
    if (buttonState == true && digitalRead(wipeB) == HIGH) {    // If button still be pressed, wipe EEPROM
      Serial.println(F("Starting Wiping EEPROM"));
      for (uint16_t x = 0; x < EEPROM.length(); x = x + 1) {    //Loop end of EEPROM address
        if (EEPROM.read(x) == 0) {              //If EEPROM address 0
          // do nothing, already clear, go to the next address in order to save time and reduce writes to EEPROM
        }
        else {
          EEPROM.write(x, 0);       // if not write 0 to clear, it takes 3.3mS
        }
      }
      Serial.println(F("EEPROM Successfully Wiped"));
      //digitalWrite(redLed, LED_OFF);  // visualize a successful wipe
      pixels.setPixelColor(0, pixels.Color(0, 0, 0)); // led off.
      pixels.show(); // This sends the updated pixel color to the hardware.
      delay(200);

      //digitalWrite(redLed, LED_ON);
      pixels.setPixelColor(0, pixels.Color(220, 0, 0)); // red color.
      pixels.show(); // This sends the updated pixel color to the hardware.
      delay(200);

      //digitalWrite(redLed, LED_OFF);
      pixels.setPixelColor(0, pixels.Color(0, 0, 0)); // led off.
      pixels.show(); // This sends the updated pixel color to the hardware.
      delay(200);

      //digitalWrite(redLed, LED_ON);
      pixels.setPixelColor(0, pixels.Color(220, 0, 0)); // red color.
      pixels.show(); // This sends the updated pixel color to the hardware.
      delay(200);

      //digitalWrite(redLed, LED_OFF);
      pixels.setPixelColor(0, pixels.Color(0, 0, 0)); // led off.
      pixels.show(); // This sends the updated pixel color to the hardware.
    }
    else {
      Serial.println(F("Wiping Cancelled")); // Show some feedback that the wipe button did not pressed for 15 seconds
      //digitalWrite(redLed, LED_OFF);
      pixels.setPixelColor(0, pixels.Color(0, 0, 0)); // led off.
      pixels.show(); // This sends the updated pixel color to the hardware.
    }
  }
  // Check if master card defined, if not let user choose a master card
  // This also useful to just redefine the Master Card
  // You can keep other EEPROM records just write other than 143 to EEPROM address 1
  // EEPROM address 1 should hold magical number which is '143'
  if (EEPROM.read(1) != 143) {
    Serial.println(F("No Master Card Defined"));
    Serial.println(F("Scan A PICC to Define as Master Card"));
    do {
      successRead = getID();            // sets successRead to 1 when we get read from reader otherwise 0
      //digitalWrite(blueLed, LED_ON);    // Visualize Master Card need to be defined
      pixels.setPixelColor(0, pixels.Color(0, 0, 220)); // blue color.
      pixels.show(); // This sends the updated pixel color to the hardware.
      delay(200);

      //digitalWrite(blueLed, LED_OFF);
      pixels.setPixelColor(0, pixels.Color(0, 0, 0)); // led off.
      pixels.show(); // This sends the updated pixel color to the hardware.
      delay(200);
    }
    while (!successRead);                  // Program will not go further while you not get a successful read
    for ( uint8_t j = 0; j < 4; j++ ) {        // Loop 4 times
      EEPROM.write( 2 + j, readCard[j] );  // Write scanned PICC's UID to EEPROM, start from address 3
    }
    EEPROM.write(1, 143);                  // Write to EEPROM we defined Master Card.
    Serial.println(F("Master Card Defined"));
  }
  Serial.println(F("-------------------"));
  Serial.println(F("Master Card's UID"));
  for ( uint8_t i = 0; i < 4; i++ ) {          // Read Master Card's UID from EEPROM
    masterCard[i] = EEPROM.read(2 + i);    // Write it to masterCard
    Serial.print(masterCard[i], HEX);
  }
  Serial.println("");
  Serial.println(F("-------------------"));
  Serial.println(F("Everything is ready"));
  Serial.println(F("Waiting PICCs to be scanned"));
  cycleLeds();    // Everything ready lets give user some feedback by cycling leds
}


///////////////////////////////////////// Main Loop ///////////////////////////////////
void loop () {

  do {

    successRead = getID();  // sets successRead to 1 when we get read from reader otherwise 0

    //When device is in use if wipe button pressed for 10 seconds initialize Master Card wiping
    if (digitalRead(wipeB) == HIGH) { // Check if button is pressed
      // Visualize normal operation is iterrupted by pressing wipe button Red is like more Warning to user
      //digitalWrite(redLed, LED_ON);  // Make sure led is on
      //digitalWrite(greenLed, LED_OFF);  // Make sure led is off
      //digitalWrite(blueLed, LED_OFF); // Make sure led is off
      pixels.setPixelColor(0, pixels.Color(220, 0, 0)); // red color.
      pixels.show(); // This sends the updated pixel color to the hardware.

      // Give some feedback
      Serial.println(F("Wipe Button Pressed"));
      Serial.println(F("Master Card will be Erased! in 10 seconds"));
      bool buttonState = monitorWipeButton(10000); // Give user enough time to cancel operation
      if (buttonState == true && digitalRead(wipeB) == HIGH) {    // If button still be pressed, wipe EEPROM
        EEPROM.write(1, 0);                  // Reset Magic Number.
        Serial.println(F("Master Card Erased from device"));
        Serial.println(F("Please reset to re-program Master Card"));
        while (1);
      }
      Serial.println(F("Master Card Erase Cancelled"));
    }


    // Sonar setup

    //Sonar Calculation distance
    long duration, distance;
    digitalWrite(sonarTrigPin, LOW);
    delayMicroseconds(2);
    digitalWrite(sonarTrigPin, HIGH);
    delayMicroseconds(10);
    digitalWrite(sonarTrigPin, LOW);
    duration = pulseIn(sonarEchoPin, HIGH);
    distance = (duration / 2) / 29.1;

    //Serial Feedback
    //if (distance >= 80 || distance <= 0) {
    //  Serial.print(distance);
    //  Serial.println("cm");
    //  Serial.println("OutOfRange");

    //}
    //else {
    //  Serial.print(distance);
    //  Serial.println(" cm");
    //}
    //delay(100);


    // LEDs Feedback
    if (distance < 80) {  // definition of distance
      pixels.setPixelColor(1, pixels.Color(0, 220, 0)); // led#1 green color.
      pixels.setPixelColor(5, pixels.Color(0, 220, 0)); // led#1 green color.
      pixels.show(); // This sends the updated pixel color to the hardware.
      delay(200); //
      pixels.setPixelColor(1, pixels.Color(0, 0, 0)); // led#1 off.
      pixels.setPixelColor(5, pixels.Color(0, 0, 0)); // led#5 off.
      pixels.setPixelColor(2, pixels.Color(0, 220, 0)); // led#2 green color.
      pixels.setPixelColor(6, pixels.Color(0, 220, 0)); // led#6 green color.
      pixels.show(); // This sends the updated pixel color to the hardware.
      delay(200); //
      pixels.setPixelColor(2, pixels.Color(0, 0, 0)); // led#2 off.
      pixels.setPixelColor(6, pixels.Color(0, 0, 0)); // led#6 off.
      pixels.setPixelColor(3, pixels.Color(0, 220, 0)); // led#3 green color.
      pixels.setPixelColor(7, pixels.Color(0, 220, 0)); // led#7 green color.
      pixels.show(); // This sends the updated pixel color to the hardware.
      delay(200); //
      pixels.setPixelColor(3, pixels.Color(0, 0, 0)); // led#3 off.
      pixels.setPixelColor(7, pixels.Color(0, 0, 0)); // led#7 off.
      pixels.setPixelColor(4, pixels.Color(0, 220, 0)); // led#4 green color.
      pixels.setPixelColor(8, pixels.Color(0, 220, 0)); // led#8 green color.
      pixels.show(); // This sends the updated pixel color to the hardware.
      delay(200); //
      pixels.setPixelColor(4, pixels.Color(0, 0, 0)); // led#4 off.
      pixels.setPixelColor(8, pixels.Color(0, 0, 0)); // led#8 off.
      pixels.setPixelColor(3, pixels.Color(0, 220, 0)); // led#3 green color.
      pixels.setPixelColor(7, pixels.Color(0, 220, 0)); // led#7 green color.
      pixels.show(); // This sends the updated pixel color to the hardware.
      delay(200); //
      pixels.setPixelColor(3, pixels.Color(0, 0, 0)); // led#3 off.
      pixels.setPixelColor(7, pixels.Color(0, 0, 0)); // led#7 off.
      pixels.setPixelColor(2, pixels.Color(0, 220, 0)); // led#2 green color.
      pixels.setPixelColor(6, pixels.Color(0, 220, 0)); // led#6 green color.
      pixels.show(); // This sends the updated pixel color to the hardware.
      delay(200); //
      pixels.setPixelColor(2, pixels.Color(0, 0, 0)); // led#2 off.
      pixels.setPixelColor(6, pixels.Color(0, 0, 0)); // led#6 off.
      pixels.setPixelColor(1, pixels.Color(0, 220, 0)); // led#1 green color.
      pixels.setPixelColor(5, pixels.Color(0, 220, 0)); // led#5 green color.
      pixels.show(); // This sends the updated pixel color to the hardware.
      delay(200); //
      pixels.setPixelColor(1, pixels.Color(0, 0, 0)); // led#2 off.
      pixels.setPixelColor(5, pixels.Color(0, 0, 0)); // led#6 off.
      pixels.setPixelColor(0, pixels.Color(0, 220, 0)); // led#0 on.
      pixels.show(); // This sends the updated pixel color to the hardware.
      delay(1000); //
      pixels.setPixelColor(0, pixels.Color(0, 0, 0)); // led#0 on.
      pixels.show(); // This sends the updated pixel color to the hardware.


    }
    else {
      pixels.setPixelColor(1, pixels.Color(0, 0, 0)); // led#1 off.
      pixels.setPixelColor(2, pixels.Color(0, 0, 0)); // led#1 off.
      pixels.setPixelColor(3, pixels.Color(0, 0, 0)); // led#1 off.
      pixels.setPixelColor(4, pixels.Color(0, 0, 0)); // led#1 off.
      pixels.setPixelColor(5, pixels.Color(0, 0, 0)); // led#1 off.
      pixels.setPixelColor(6, pixels.Color(0, 0, 0)); // led#1 off.
      pixels.setPixelColor(7, pixels.Color(0, 0, 0)); // led#1 off.
      pixels.setPixelColor(8, pixels.Color(0, 0, 0)); // led#1 off.
      pixels.show(); // This sends the updated pixel color to the hardware.
    }



    //Check whether the learning kit is back on the shelf
    switchState = digitalRead(2); // check what is the state of the button detecting the learning kit
    if ( switchState == 1) { // check if the button detecting the learning kit is pressed, if it is :
      Serial.println(F("Learning kit is detected"));
      delay(1000);
      myservo.write(90); // locks the learning kit
      delay(15);
    }
    if (programMode) {
      cycleLeds();              // Program Mode cycles through Red Green Blue waiting to read a new card
    }
    else {
      normalModeOn();     // Normal mode, blue Power LED is off
    }
  }
  while (!successRead);   //the program will not go further while you are not getting a successful read
  if (programMode) {
    if ( isMaster(readCard) ) { //When in program mode check First If master card scanned again to exit program mode
      Serial.println(F("Master Card Scanned"));
      Serial.println(F("Exiting Program Mode"));
      Serial.println(F("-----------------------------"));
      programMode = false;
      return;
    }
    else {
      if ( findID(readCard) ) { // If scanned card is known delete it
        Serial.println(F("I know this PICC, removing..."));
        deleteID(readCard);
        Serial.println("-----------------------------");
        Serial.println(F("Scan a PICC to ADD or REMOVE to EEPROM"));
      }
      else {                    // If scanned card is not known add it
        Serial.println(F("I do not know this PICC, adding..."));
        writeID(readCard);
        Serial.println(F("-----------------------------"));
        Serial.println(F("Scan a PICC to ADD or REMOVE to EEPROM"));
      }
    }
  }
  else {
    if ( isMaster(readCard)) {    // If scanned card's ID matches Master Card's ID - enter program mode
      programMode = true;
      Serial.println(F("Hello Master - Entered Program Mode"));
      uint8_t count = EEPROM.read(0);   // Read the first Byte of EEPROM that
      Serial.print(F("I have "));     // stores the number of ID's in EEPROM
      Serial.print(count);
      Serial.print(F(" record(s) on EEPROM"));
      Serial.println("");
      Serial.println(F("Scan a PICC to ADD or REMOVE to EEPROM"));
      Serial.println(F("Scan Master Card again to Exit Program Mode"));
      Serial.println(F("-----------------------------"));
    }
    else {
      if ( findID(readCard) ) { // If not, see if the card is in the EEPROM
        Serial.println(F("Learning Kit unlocked, enjoy!"));
        granted(300);         // Open the door lock for 300 ms
      }
      else {      // If not, show that the ID was not valid
        Serial.println(F("Talk to an Instructor to access this Learning Kit, Thanks."));
        denied();
      }
    }
  }
}

/////////////////////////////////////////  Access Granted    ///////////////////////////////////
void granted ( uint16_t setDelay) {
  //digitalWrite(blueLed, LED_OFF);   // Turn off blue LED
  //digitalWrite(redLed, LED_OFF);  // Turn off red LED
  //digitalWrite(greenLed, LED_ON);   // Turn on green LED
  pixels.setPixelColor(0, pixels.Color(0, 220, 0)); // green color.
  pixels.show(); // This sends the updated pixel color to the hardware.

  //delay(1000);
  myservo.write(0); // Unlock learning kit!
  delay(5000); // Hold learning kit unlocked for given seconds

  //Check whether the learning kit is still on the shelf
  switchState = digitalRead(2); // check what is the state of the button detecting the learning kit
  if ( switchState == 1) { // check if the button detecting the learning kit is pressed, if it is :
    Serial.println(F("Learning kit is still detected"));
    delay(1000);
    myservo.write(90); // locks the learning kit
    delay(15);
  }

  //digitalWrite(relay, LOW);     // Unlock door!
  //delay(setDelay);          // Hold door lock open for given seconds
  //digitalWrite(relay, HIGH);    // Relock door
  delay(1000);            // Hold green LED on for a second
}

///////////////////////////////////////// Access Denied  ///////////////////////////////////
void denied() {
  //digitalWrite(greenLed, LED_OFF);  // Make sure green LED is off
  //digitalWrite(blueLed, LED_OFF);   // Make sure blue LED is off
  //digitalWrite(redLed, LED_ON);   // Turn on red LED
  pixels.setPixelColor(0, pixels.Color(220, 0, 0)); // red color.
  pixels.show(); // This sends the updated pixel color to the hardware.
  delay(2000);
}


///////////////////////////////////////// Get PICC's UID ///////////////////////////////////
uint8_t getID() {
  // Getting ready for Reading PICCs
  if ( ! mfrc522.PICC_IsNewCardPresent()) { //If a new PICC placed to RFID reader continue
    return 0;
  }
  if ( ! mfrc522.PICC_ReadCardSerial()) {   //Since a PICC placed get Serial and continue
    return 0;
  }
  // There are Mifare PICCs which have 4 byte or 7 byte UID care if you use 7 byte PICC
  // I think we should assume every PICC as they have 4 byte UID
  // Until we support 7 byte PICCs
  Serial.println(F("Scanned PICC's UID:"));
  for ( uint8_t i = 0; i < 4; i++) {  //
    readCard[i] = mfrc522.uid.uidByte[i];
    Serial.print(readCard[i], HEX);
  }
  Serial.println("");
  mfrc522.PICC_HaltA(); // Stop reading
  return 1;
}

void ShowReaderDetails() {
  // Get the MFRC522 software version
  byte v = mfrc522.PCD_ReadRegister(mfrc522.VersionReg);
  Serial.print(F("MFRC522 Software Version: 0x"));
  Serial.print(v, HEX);
  if (v == 0x91)
    Serial.print(F(" = v1.0"));
  else if (v == 0x92)
    Serial.print(F(" = v2.0"));
  else
    Serial.print(F(" (unknown),probably a chinese clone?"));
  Serial.println("");
  // When 0x00 or 0xFF is returned, communication probably failed
  if ((v == 0x00) || (v == 0xFF)) {
    Serial.println(F("WARNING: Communication failure, is the MFRC522 properly connected?"));
    Serial.println(F("SYSTEM HALTED: Check connections."));
    // Visualize system is halted
    //digitalWrite(greenLed, LED_OFF);  // Make sure green LED is off
    //digitalWrite(blueLed, LED_OFF);   // Make sure blue LED is off
    //digitalWrite(redLed, LED_ON);   // Turn on red LED
    pixels.setPixelColor(0, pixels.Color(220, 0, 0)); // red color.
    pixels.show(); // This sends the updated pixel color to the hardware.
    while (true); // do not go further
  }
}

///////////////////////////////////////// Cycle Leds (Program Mode) ///////////////////////////////////
void cycleLeds() {
  //digitalWrite(redLed, LED_OFF);  // Make sure red LED is off
  //digitalWrite(greenLed, LED_ON);   // Make sure green LED is on
  //digitalWrite(blueLed, LED_OFF);   // Make sure blue LED is off
  pixels.setPixelColor(0, pixels.Color(0, 220, 0)); // green color.
  pixels.show(); // This sends the updated pixel color to the hardware.
  delay(200);
  //digitalWrite(redLed, LED_OFF);  // Make sure red LED is off
  //digitalWrite(greenLed, LED_OFF);  // Make sure green LED is off
  //digitalWrite(blueLed, LED_ON);  // Make sure blue LED is on
  pixels.setPixelColor(0, pixels.Color(0, 0, 220)); // blue color.
  pixels.show(); // This sends the updated pixel color to the hardware.
  delay(200);
  //digitalWrite(redLed, LED_ON);   // Make sure red LED is on
  //digitalWrite(greenLed, LED_OFF);  // Make sure green LED is off
  //digitalWrite(blueLed, LED_OFF);   // Make sure blue LED is off
  pixels.setPixelColor(0, pixels.Color(220, 0, 0)); // blue color.
  pixels.show(); // This sends the updated pixel color to the hardware.
  delay(200);
}

//////////////////////////////////////// Normal Mode Led  ///////////////////////////////////
void normalModeOn () {
  //digitalWrite(blueLed, LED_ON);  // Blue LED ON and ready to read card
  //digitalWrite(redLed, LED_OFF);  // Make sure Red LED is off
  //digitalWrite(greenLed, LED_OFF);  // Make sure Green LED is off
  pixels.setPixelColor(0, pixels.Color(0, 0, 220)); // blue color.
  pixels.show(); // This sends the updated pixel color to the hardware.
  //digitalWrite(relay, HIGH);    // Make sure Door is Locked


}

//////////////////////////////////////// Read an ID from EEPROM //////////////////////////////
void readID( uint8_t number ) {
  uint8_t start = (number * 4 ) + 2;    // Figure out starting position
  for ( uint8_t i = 0; i < 4; i++ ) {     // Loop 4 times to get the 4 Bytes
    storedCard[i] = EEPROM.read(start + i);   // Assign values read from EEPROM to array
  }
}

///////////////////////////////////////// Add ID to EEPROM   ///////////////////////////////////
void writeID( byte a[] ) {
  if ( !findID( a ) ) {     // Before we write to the EEPROM, check to see if we have seen this card before!
    uint8_t num = EEPROM.read(0);     // Get the numer of used spaces, position 0 stores the number of ID cards
    uint8_t start = ( num * 4 ) + 6;  // Figure out where the next slot starts
    num++;                // Increment the counter by one
    EEPROM.write( 0, num );     // Write the new count to the counter
    for ( uint8_t j = 0; j < 4; j++ ) {   // Loop 4 times
      EEPROM.write( start + j, a[j] );  // Write the array values to EEPROM in the right position
    }
    successWrite();
    Serial.println(F("Succesfully added ID record to EEPROM"));
  }
  else {
    failedWrite();
    Serial.println(F("Failed! There is something wrong with ID or bad EEPROM"));
  }
}

///////////////////////////////////////// Remove ID from EEPROM   ///////////////////////////////////
void deleteID( byte a[] ) {
  if ( !findID( a ) ) {     // Before we delete from the EEPROM, check to see if we have this card!
    failedWrite();      // If not
    Serial.println(F("Failed! There is something wrong with ID or bad EEPROM"));
  }
  else {
    uint8_t num = EEPROM.read(0);   // Get the numer of used spaces, position 0 stores the number of ID cards
    uint8_t slot;       // Figure out the slot number of the card
    uint8_t start;      // = ( num * 4 ) + 6; // Figure out where the next slot starts
    uint8_t looping;    // The number of times the loop repeats
    uint8_t j;
    uint8_t count = EEPROM.read(0); // Read the first Byte of EEPROM that stores number of cards
    slot = findIDSLOT( a );   // Figure out the slot number of the card to delete
    start = (slot * 4) + 2;
    looping = ((num - slot) * 4);
    num--;      // Decrement the counter by one
    EEPROM.write( 0, num );   // Write the new count to the counter
    for ( j = 0; j < looping; j++ ) {         // Loop the card shift times
      EEPROM.write( start + j, EEPROM.read(start + 4 + j));   // Shift the array values to 4 places earlier in the EEPROM
    }
    for ( uint8_t k = 0; k < 4; k++ ) {         // Shifting loop
      EEPROM.write( start + j + k, 0);
    }
    successDelete();
    Serial.println(F("Succesfully removed ID record from EEPROM"));
  }
}

///////////////////////////////////////// Check Bytes   ///////////////////////////////////
bool checkTwo ( byte a[], byte b[] ) {
  for ( uint8_t k = 0; k < 4; k++ ) {   // Loop 4 times
    if ( a[k] != b[k] ) {     // IF a != b then false, because: one fails, all fail
      return false;
    }
  }
  return true;
}

///////////////////////////////////////// Find Slot   ///////////////////////////////////
uint8_t findIDSLOT( byte find[] ) {
  uint8_t count = EEPROM.read(0);       // Read the first Byte of EEPROM that
  for ( uint8_t i = 1; i <= count; i++ ) {    // Loop once for each EEPROM entry
    readID(i);                // Read an ID from EEPROM, it is stored in storedCard[4]
    if ( checkTwo( find, storedCard ) ) {   // Check to see if the storedCard read from EEPROM
      // is the same as the find[] ID card passed
      return i;         // The slot number of the card
    }
  }
}

///////////////////////////////////////// Find ID From EEPROM   ///////////////////////////////////
bool findID( byte find[] ) {
  uint8_t count = EEPROM.read(0);     // Read the first Byte of EEPROM that
  for ( uint8_t i = 1; i < count; i++ ) {    // Loop once for each EEPROM entry
    readID(i);          // Read an ID from EEPROM, it is stored in storedCard[4]
    if ( checkTwo( find, storedCard ) ) {   // Check to see if the storedCard read from EEPROM
      return true;
    }
    else {    // If not, return false
    }
  }
  return false;
}

///////////////////////////////////////// Write Success to EEPROM   ///////////////////////////////////
// Flashes the green LED 3 times to indicate a successful write to EEPROM
void successWrite() {

  //digitalWrite(blueLed, LED_OFF);   // Make sure blue LED is off
  //digitalWrite(redLed, LED_OFF);  // Make sure red LED is off
  //digitalWrite(greenLed, LED_OFF);  // Make sure green LED is on
  pixels.setPixelColor(0, pixels.Color(0, 0, 0)); // led off.
  pixels.show(); // This sends the updated pixel color to the hardware.

  delay(200);

  //digitalWrite(greenLed, LED_ON);   // Make sure green LED is on
  pixels.setPixelColor(0, pixels.Color(0, 220, 0)); // green color.
  pixels.show(); // This sends the updated pixel color to the hardware.

  delay(200);

  //digitalWrite(greenLed, LED_OFF);  // Make sure green LED is off
  pixels.setPixelColor(0, pixels.Color(0, 0, 0)); // led off.
  pixels.show(); // This sends the updated pixel color to the hardware.

  delay(200);

  //digitalWrite(greenLed, LED_ON);   // Make sure green LED is on
  pixels.setPixelColor(0, pixels.Color(0, 220, 0)); // green color.
  pixels.show(); // This sends the updated pixel color to the hardware.

  delay(200);

  //digitalWrite(greenLed, LED_OFF);  // Make sure green LED is off
  pixels.setPixelColor(0, pixels.Color(0, 0, 0)); // led off.
  pixels.show(); // This sends the updated pixel color to the hardware.

  delay(200);

  //digitalWrite(greenLed, LED_ON);   // Make sure green LED is on
  pixels.setPixelColor(0, pixels.Color(0, 220, 0)); // green color.
  pixels.show(); // This sends the updated pixel color to the hardware.

  delay(200);
}

///////////////////////////////////////// Write Failed to EEPROM   ///////////////////////////////////
// Flashes the red LED 3 times to indicate a failed write to EEPROM
void failedWrite() {
  //digitalWrite(blueLed, LED_OFF);   // Make sure blue LED is off
  //digitalWrite(redLed, LED_OFF);  // Make sure red LED is off
  //digitalWrite(greenLed, LED_OFF);  // Make sure green LED is off
  pixels.setPixelColor(0, pixels.Color(0, 0, 0)); // led off.
  pixels.show(); // This sends the updated pixel color to the hardware.

  delay(200);

  //digitalWrite(redLed, LED_ON);   // Make sure red LED is on
  pixels.setPixelColor(0, pixels.Color(220, 0, 0)); // red color.
  pixels.show(); // This sends the updated pixel color to the hardware.

  delay(200);

  //digitalWrite(redLed, LED_OFF);  // Make sure red LED is off
  pixels.setPixelColor(0, pixels.Color(0, 0, 0)); // led off.
  pixels.show(); // This sends the updated pixel color to the hardware.

  delay(200);

  //digitalWrite(redLed, LED_ON);   // Make sure red LED is on
  pixels.setPixelColor(0, pixels.Color(220, 0, 0)); // red color.
  pixels.show(); // This sends the updated pixel color to the hardware.

  delay(200);

  //digitalWrite(redLed, LED_OFF);  // Make sure red LED is off
  pixels.setPixelColor(0, pixels.Color(0, 0, 0)); // led off.
  pixels.show(); // This sends the updated pixel color to the hardware.

  delay(200);

  //digitalWrite(redLed, LED_ON);   // Make sure red LED is on
  pixels.setPixelColor(0, pixels.Color(220, 0, 0)); // red color.
  pixels.show(); // This sends the updated pixel color to the hardware.

  delay(200);
}

///////////////////////////////////////// Success Remove UID From EEPROM  ///////////////////////////////////
// Flashes the blue LED 3 times to indicate a success delete to EEPROM
void successDelete() {

  //digitalWrite(blueLed, LED_OFF);   // Make sure blue LED is off
  //digitalWrite(redLed, LED_OFF);  // Make sure red LED is off
  //digitalWrite(greenLed, LED_OFF);  // Make sure green LED is off
  pixels.setPixelColor(0, pixels.Color(0, 0, 0)); // led off.
  pixels.show(); // This sends the updated pixel color to the hardware.
  delay(200);

  //digitalWrite(blueLed, LED_ON);  // Make sure blue LED is on
  pixels.setPixelColor(0, pixels.Color(0, 0, 220)); // blue color.
  pixels.show(); // This sends the updated pixel color to the hardware.
  delay(200);

  //digitalWrite(blueLed, LED_OFF);   // Make sure blue LED is off
  pixels.setPixelColor(0, pixels.Color(0, 0, 0)); // led off.
  pixels.show(); // This sends the updated pixel color to the hardware.

  delay(200);

  //digitalWrite(blueLed, LED_ON);  // Make sure blue LED is on
  pixels.setPixelColor(0, pixels.Color(0, 0, 220)); // blue color.
  pixels.show(); // This sends the updated pixel color to the hardware.
  delay(200);

  //digitalWrite(blueLed, LED_OFF);   // Make sure blue LED is off
  pixels.setPixelColor(0, pixels.Color(0, 0, 0)); // led off.
  pixels.show(); // This sends the updated pixel color to the hardware.
  delay(200);

  //digitalWrite(blueLed, LED_ON);  // Make sure blue LED is on
  pixels.setPixelColor(0, pixels.Color(0, 0, 220)); // blue color.
  pixels.show(); // This sends the updated pixel color to the hardware.
  delay(200);
}

////////////////////// Check readCard IF is masterCard   ///////////////////////////////////
// Check to see if the ID passed is the master programing card
bool isMaster( byte test[] ) {
  return checkTwo(test, masterCard);
}

bool monitorWipeButton(uint32_t interval) {
  uint32_t now = (uint32_t)millis();
  while ((uint32_t)millis() - now < interval)  {
    // check on every half a second
    if (((uint32_t)millis() % 500) == 0) {
      if (digitalRead(wipeB) != HIGH)
        return false;
    }
  }
  return true;
}
