#include <movingAvg.h>
#include <Servo.h>

/*
   PINOUT
*/
// pin 0 RX
// pin 1 TX
#define dcmotorA 5  // pin dc motor A
#define dcmotorB 6 // pin dc motor B
#define playServo 9 // pin servo
#define dir_stepperAzim 2 // pin direction stepper A
#define step_stepperAzim 3 // pin step stepper A
#define pot_stepperAzim A0 // pin potentiometer A
#define dir_stepperEle 7 // pin direction stepper B
#define step_stepperEle 8 // pin step stepper B
#define pot_stepperEle A1 // pin potentiometer B

#define azimuth_pot_tolerance 1 // max diff accepted between azimuth pot sensor position and requested azimuth position
#define elevation_pot_tolerance 1 // max diff accepted between elevation pot sensor position and requested elevation position
#define azim_max_travel 120 // max angle azimuth stepper
#define ele_max_travel 60 // max angle elevation stepper
#define stepper_speed 1 // delay between the pulses, increase the value decrease the speed

#define stowed_pusher_servo_pos 160 // default position for the ball pusher
#define deploy_pusher_servo_pos 4 // full deply servo position


Servo myservo;  // create servo object to control a servo
movingAvg potReadings(10);  // create moving average object with 10 data points

// set speed of dc motor A 0-255
void set_dcaspeed(int speed) {
  if (speed > 0) { 
    analogWrite(dcmotorA, map(speed, 1, 100, 40, 255));
  } else {
    analogWrite(dcmotorA, 0);
  }
}

// set speed of dc motor B 0-255
void set_dcbspeed(int speed) {
  if (speed > 0) { 
    analogWrite(dcmotorB, map(speed, 1, 100, 40, 255));
  } else {
    analogWrite(dcmotorB, 0);
  }
}

// launches the ball
void play_servo() {
  myservo.write(deploy_pusher_servo_pos);
  delay(1000);
  myservo.write(stowed_pusher_servo_pos);
}

// input string from serial
String inputString = "";         // a String to hold incoming data
boolean stringComplete = false;  // whether the string is complete

// move stepper wrapper for azimuth
void move_azimuth(int req_pos) {
  move_stepper(req_pos, pot_stepperAzim, step_stepperAzim, dir_stepperAzim, azim_max_travel, azimuth_pot_tolerance);
}

// move stepper wrapper for elevation
void move_elevation(int req_pos) {
  move_stepper(req_pos, pot_stepperEle, step_stepperEle, dir_stepperEle, ele_max_travel, elevation_pot_tolerance);
}

// reach requested stepper position
void move_stepper(int req_pos, int pot_pin, int step_pin, int dir_pin, int pos_max, int tol) {
  if (req_pos > pos_max) {
    req_pos = pos_max;
  } else if(req_pos < 0) {
    req_pos = 0;
  }
  
  for (int i=0; i<10; i++) {
    potReadings.reading(analogRead(pot_pin));
  }

  int potMapped = map(potReadings.getAvg(), 0, 1023, 0, pos_max);
  Serial.print("pot: ");
  Serial.println(potMapped);
  
  while ((potMapped < req_pos-tol)||(potMapped > req_pos+tol)) {
    if (req_pos < potMapped) {
      digitalWrite(dir_pin, HIGH);
    } else {
      digitalWrite(dir_pin, LOW);
    }
    digitalWrite(step_pin, HIGH);
    digitalWrite(step_pin, LOW);
     
    potReadings.reading(analogRead(pot_pin));
    potMapped = map(potReadings.getAvg(), 0, 1023, 0, pos_max);
    delay(stepper_speed);
  }
}


/*
   SETUP
*/
void setup() {

  // initialize serial:
  Serial.begin(9600);
  // reserve 200 bytes for the inputString:
  inputString.reserve(200);

  // set motor pins as ouput
  pinMode(dcmotorA, OUTPUT);
  set_dcaspeed(0);
  pinMode(dcmotorB, OUTPUT);
  set_dcaspeed(0);

  // initialize the servo
  myservo.attach(playServo);  // attaches the servo on pin 11 to the servo object
  myservo.write(stowed_pusher_servo_pos);

  // initialize rolling average
  potReadings.begin();
}

/*
   LOOP
*/
void loop() {

  // print the string when a newline arrives:
  if (stringComplete) {
    
    int value = inputString.substring(1).toInt();
    
    switch (inputString[0]) {
      case 'A':
        set_dcaspeed(value);
        break;
      case 'B':
        set_dcbspeed(value);
        break;
      case 'C':
        play_servo();
        break;
      case 'D':
        move_azimuth(value);
        break;
      case 'E':
        move_elevation(value);
        break;
    }

    // clear the string:
    inputString = "";
    stringComplete = false;
  }


}

/*
  SerialEvent occurs whenever a new data comes in the hardware serial RX. This
  routine is run between each time loop() runs, so using delay inside loop can
  delay response. Multiple bytes of data may be available.
*/
void serialEvent() {
  while (Serial.available()) {
    // get the new byte:
    char inChar = (char)Serial.read();
    // if the incoming character is a newline, set a flag so the main loop can
    // do something about it:
    if (inChar == '\n') {
      stringComplete = true;
    }
    else {
      // add it to the inputString:
      inputString += inChar;
    }
  }
}
