//
// hello.ftdi.io
//
// 9600 baud FTDI serial io, interrupt version
// Push-button input, interrupt version
// LED output for characters 0 and 1
//
// set lfuse to 0x5E for 20 MHz xtal
//
// Axel Cornu
// 15 March 2019
//

#include <avr/io.h>
#include <util/delay.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>

#define output(directions,pin) (directions |= pin) // set port direction for output
#define set(port,pin) (port |= pin) // set port pin
#define clear(port,pin) (port &= (~pin)) // clear port pin
#define pin_test(pins,pin) (pins & pin) // test for port pin
#define bit_test(byte,bit) (byte & (1 << bit)) // test for bit set
#define bit_delay_time 102 // bit delay for 9600 with overhead
#define bit_delay() _delay_us(bit_delay_time) // RS232 bit delay
#define half_bit_delay() _delay_us(bit_delay_time/2) // RS232 half bit delay
#define char_delay() _delay_ms(10) // char delay

#define serial_port PORTA
#define serial_direction DDRA
#define serial_pins PINA
#define serial_pin_in (1 << PA0)
#define serial_pin_out (1 << PA1)
#define serial_interrupt (1 << PCIE0)	// GIMSK
#define all_interrupts 0x50
#define pc_interrupts_pin (1 << PCINT0)		// PCMSK0

#define LED0 (1 << PA2)
#define LED1 (1 << PA3)

#define max_buffer 25

volatile char flag = 0;

void get_char(volatile unsigned char *pins, unsigned char pin, char *rxbyte) {
   //
   // read character into rxbyte on pins pin
   //    assumes line driver (inverts bits)
   //
   *rxbyte = 0;
   //
   // delay to middle of first data bit
   //
   half_bit_delay();
   bit_delay();
   //
   // unrolled loop to read data bits
   //
   if pin_test(*pins,pin)
      *rxbyte |= (1 << 0);
   else
      *rxbyte |= (0 << 0);
   bit_delay();
   if pin_test(*pins,pin)
      *rxbyte |= (1 << 1);
   else
      *rxbyte |= (0 << 1);
   bit_delay();
   if pin_test(*pins,pin)
      *rxbyte |= (1 << 2);
   else
      *rxbyte |= (0 << 2);
   bit_delay();
   if pin_test(*pins,pin)
      *rxbyte |= (1 << 3);
   else
      *rxbyte |= (0 << 3);
   bit_delay();
   if pin_test(*pins,pin)
      *rxbyte |= (1 << 4);
   else
      *rxbyte |= (0 << 4);
   bit_delay();
   if pin_test(*pins,pin)
      *rxbyte |= (1 << 5);
   else
      *rxbyte |= (0 << 5);
   bit_delay();
   if pin_test(*pins,pin)
      *rxbyte |= (1 << 6);
   else
      *rxbyte |= (0 << 6);
   bit_delay();
   if pin_test(*pins,pin)
      *rxbyte |= (1 << 7);
   else
      *rxbyte |= (0 << 7);
   //
   // wait for stop bit
   //
   bit_delay();
   half_bit_delay();
   }

void put_char(volatile unsigned char *port, unsigned char pin, char txchar) {
   //
   // send character in txchar on port pin
   //    assumes line driver (inverts bits)
   //
   // start bit
   //
   clear(*port,pin);
   bit_delay();
   //
   // unrolled loop to write data bits
   //
   if bit_test(txchar,0)
      set(*port,pin);
   else
      clear(*port,pin);
   bit_delay();
   if bit_test(txchar,1)
      set(*port,pin);
   else
      clear(*port,pin);
   bit_delay();
   if bit_test(txchar,2)
      set(*port,pin);
   else
      clear(*port,pin);
   bit_delay();
   if bit_test(txchar,3)
      set(*port,pin);
   else
      clear(*port,pin);
   bit_delay();
   if bit_test(txchar,4)
      set(*port,pin);
   else
      clear(*port,pin);
   bit_delay();
   if bit_test(txchar,5)
      set(*port,pin);
   else
      clear(*port,pin);
   bit_delay();
   if bit_test(txchar,6)
      set(*port,pin);
   else
      clear(*port,pin);
   bit_delay();
   if bit_test(txchar,7)
      set(*port,pin);
   else
      clear(*port,pin);
   bit_delay();
   //
   // stop bit
   //
   set(*port,pin);
   bit_delay();
   }

void put_string(volatile unsigned char *port, unsigned char pin, char *str) {
   //
   // print a null-terminated string
   //
   static int index;
   index = 0;
   do {
      put_char(port, pin, str[index]);
      ++index;
      } while (str[index] != 0);
   }

ISR(PCINT0_vect) {
   //
   // pin change interrupt handler
   //
   static char chr;

   get_char(&serial_pins, serial_pin_in, &chr);
   if (chr > 32) {
   put_string(&serial_port, serial_pin_out, "You typed ");

   put_char(&serial_port, serial_pin_out, chr);
   put_char(&serial_port, serial_pin_out, 10); // new line
   		if (chr == 48) {	// number 0
			clear(serial_port,LED1);
			set(serial_port,LED0);
		}
		else if (chr == 49) {	// number 1
			clear(serial_port,LED0);
			set(serial_port,LED1);
		}
   }
}

ISR (INT0_vect) // external interrupt_zero ISR
{
	flag = 1;
}

int main(void) {
   //
   // main
   //
   cli(); //disable global interrupts
   //
   // set clock divider to /1
   //
   CLKPR = (1 << CLKPCE);
   CLKPR = (0 << CLKPS3) | (0 << CLKPS2) | (0 << CLKPS1) | (0 << CLKPS0);
   //
   // initialize output pins
   //
   set(serial_port, serial_pin_out);
   output(serial_direction, serial_pin_out);
   output(serial_direction, LED0);
   output(serial_direction, LED1);
   clear(serial_port,LED0);
   clear(serial_port,LED1);
   // external interrupt(s) initialization
   // INT0: On
   // INT0 Mode: Falling Edge
   MCUCR=0x02;
   GIFR=0x40;

   //
   // set up pin change interrupt on input pin
   //
   set(GIMSK, all_interrupts);
   set (PCMSK0, pc_interrupts_pin);
   sei();
   //
   // main loop
   //
   while (1) {
      //
      // wait for interrupt
      //
	  if(flag==1){
		put_string(&serial_port, serial_pin_out, "Pushed !");
		put_char(&serial_port, serial_pin_out, 10); // new line
		set(GIMSK, serial_interrupt);
		_delay_ms(200);	// debounce
		set(GIMSK, all_interrupts);
		flag = 0;
	  }
   }
}