#include <WiFi.h>
#include <ESPAsyncWebServer.h>
#include <ESPmDNS.h>

// --- Pin Definitions ---
#define PHASE_A_PIN 4
#define ENABLE_A_PIN 5
#define PHASE_B_PIN 6
#define ENABLE_B_PIN 7

// --- WiFi Configuration ---
static const char* WIFI_SSID = "ATTsmith";
static const char* WIFI_PASS = "rolltide9172";

// --- mDNS Configuration ---
static const char* MDNS_HOSTNAME = "maestrostepper";
static const char* CAMERA_MDNS_HOSTNAME = "maestrocamera";

// --- Global Variables ---
AsyncWebServer server(80);

int targetDirection = 0;
unsigned long lastStepTime = 0;
unsigned long step_delay = 10;
int currentStep = 0;


// Define logic levels
#define COIL_ENABLE HIGH
#define COIL_DISABLE LOW

void setup() {
  Serial.begin(115200);

  // Connect to WiFi
  WiFi.begin(WIFI_SSID, WIFI_PASS);
  Serial.print("Connecting to hamburger...");
  while (WiFi.status() != WL_CONNECTED) {
    delay(1000);
    Serial.print(".");
  }
  Serial.println("\nConnected to WiFi!");
  WiFi.setSleep(WIFI_PS_NONE);
  Serial.print("IP Address: ");
  Serial.println(WiFi.localIP());

  // Setup motor pins
  pinMode(PHASE_A_PIN, OUTPUT);
  pinMode(ENABLE_A_PIN, OUTPUT);
  pinMode(PHASE_B_PIN, OUTPUT);
  pinMode(ENABLE_B_PIN, OUTPUT);

  // mDNS
  if (!MDNS.begin(MDNS_HOSTNAME)) {
    Serial.println("Error setting up MDNS responder!");
    while (1) delay(1000);
  }
  Serial.println("mDNS responder started: http://" + String(MDNS_HOSTNAME) + ".local");
  MDNS.addService("http", "tcp", 80);

  // HTML page
server.on("/", HTTP_GET, [](AsyncWebServerRequest* request) {
String html = R"rawliteral(
<!DOCTYPE html>
<html>
<head>
  <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
  <title>Joystick Stepper Control</title>
  <script src="https://cdnjs.cloudflare.com/ajax/libs/nipplejs/0.9.0/nipplejs.min.js"></script>
  <style>
    html, body {
      margin: 0;
      padding: 0;
      background: black;
      overflow: hidden;
      width: 100%;
      height: 100%;
      touch-action: none;
      font-family: sans-serif;
    }

    #container {
      position: relative;
      width: 100vw;
      height: 100vh;
      display: flex;
      justify-content: center;
      align-items: center;
      background: black;
    }

    #camera {
      width: 100%;
      height: 100%;
      object-fit: contain; /* preserve 16:9 aspect ratio */
      background: black;
    }

    #joystick-zone {
      position: absolute;
      bottom: 5%;
      left: 5%;
      width: 25vw;
      height: 25vw;
      max-width: 150px;
      max-height: 150px;
      border-radius: 50%;
      background-color: rgba(255, 255, 255, 0.1);
      z-index: 10;
    }

    #status {
      position: absolute;
      top: 10px;
      left: 10px;
      color: white;
      font-size: 14px;
      background-color: rgba(0, 0, 0, 0.5);
      padding: 6px 10px;
      border-radius: 6px;
      z-index: 10;
    }
  </style>
</head>
<body>
  <div id="container">
    <img id="camera" src="http://maestrocamera.local/1280x720.mjpeg" alt="Camera Stream">
    <div id="joystick-zone"></div>
    <div id="status">Waiting for input...</div>
  </div>

  <script>
    const zone = document.getElementById("joystick-zone");
    const status = document.getElementById("status");

    const joystick = nipplejs.create({
      zone: zone,
      mode: "static",
      position: { left: "50%", top: "50%" },
      color: "blue",
      size: Math.min(window.innerWidth * 0.25, 150)
    });

    let lastSent = 0;

    joystick.on("move", (evt, data) => {
      if (Date.now() - lastSent > 100 && data.direction) {
        lastSent = Date.now();
        const x = Math.round(data.force * Math.cos(data.angle.radian) * 100);
        const y = Math.round(data.force * Math.sin(data.angle.radian) * 100);

        status.textContent = `X: ${x}, Y: ${y}`;
        fetch(`/move?x=${x}&y=${y}`).catch(err => console.error(err));
      }
    });

    joystick.on("end", () => {
      status.textContent = "Stopped";
      fetch("/move?x=0&y=0");
    });

    window.addEventListener("resize", () => {
      location.reload(); // re-init size if screen orientation changes
    });
  </script>
</body>
</html>
)rawliteral";
  request->send(200, "text/html", html);
});

server.on("/move", HTTP_GET, [](AsyncWebServerRequest* request) {
  if (request->hasParam("x")) {
    int x = request->getParam("x")->value().toInt();
    Serial.println(x);
    // Normalize X to -1, 0, or 1
    if (x > 20) {
      targetDirection = 1;
    } else if (x < -20) {
      targetDirection = -1;
    } else {
      targetDirection = 0;
    }

    // Map |x| (0 to 100) to step delay (20ms to 5ms)
    int absX = abs(x);
    absX = constrain(absX, 0, 100);
    step_delay = map(absX, 0, 100, 20, 5);

    request->send(200, "text/plain", "Direction and speed set");
  } else {
    request->send(400, "text/plain", "Missing x");
  }
});




  server.begin();
  Serial.println("HTTP server started");
}

void setMotorStep(int step) {
  int phase = step % 4;
  if (phase < 0) phase += 4;

  switch (phase) {
    case 0:
      digitalWrite(PHASE_A_PIN, HIGH);
      digitalWrite(ENABLE_A_PIN, HIGH);
      digitalWrite(PHASE_B_PIN, HIGH);
      digitalWrite(ENABLE_B_PIN, HIGH);
      break;
    case 1:
      digitalWrite(PHASE_A_PIN, LOW);
      digitalWrite(ENABLE_A_PIN, HIGH);
      digitalWrite(PHASE_B_PIN, HIGH);
      digitalWrite(ENABLE_B_PIN, HIGH);
      break;
    case 2:
      digitalWrite(PHASE_A_PIN, LOW);
      digitalWrite(ENABLE_A_PIN, HIGH);
      digitalWrite(PHASE_B_PIN, LOW);
      digitalWrite(ENABLE_B_PIN, HIGH);
      break;
    case 3:
      digitalWrite(PHASE_A_PIN, HIGH);
      digitalWrite(ENABLE_A_PIN, HIGH);
      digitalWrite(PHASE_B_PIN, LOW);
      digitalWrite(ENABLE_B_PIN, HIGH);
      break;
  }
}


void loop() {
  unsigned long now = millis();
  if (targetDirection != 0 && now - lastStepTime > step_delay) {
    currentStep += targetDirection;
    setMotorStep(currentStep);
    lastStepTime = now;
  }
}
